/*global define, require */
/*jslint white: true */

/*
	Node:

	This object provides a mixin for each node within the visual hierarchy.
*/

define	(["src/utils", "src/build/treeLeaf", "src/build/changeNotifier", "src/build/VisibleItem", "src/math/Mat3"],
function(utils, treeLeaf, changeNotifier, VisibleItem, mat3) {
	"use strict";

	var VisibleItem_clone = VisibleItem.prototype.clone;

	function Node(inTypeName, inNodeName) {
		this.VisibleItem("VisibleItem");

		// do not modify these private members directly or notifiers won't be called correctly
		this.pNode = { 
			name				: inNodeName, 
			typeName			: inTypeName, 
			parent				: undefined,
			matrix				: mat3().setIdentity(),
			blendMode			: null,
			bClipped			: false,
			parentCanWarpMe		: true,
			changeNotifiers		: undefined
		};

		this.pMesh = null;

		// valid after installGeometryComponent
		this.privateGeometry = {};
	}

	utils.mixin(Node,
		VisibleItem,
		treeLeaf({ 
			parentLabel   : "parent", 
			getTreeInfo : function (obj) { return obj.pNode; }
		}),
		changeNotifier({
			notifiersLabel: "changeNotifiers",
			getTreeInfo: function (obj) { return obj.pNode;	}		
		}),
		{
			getTypeName : function () {
				return this.pNode.typeName;
			},

			setName : function (name) {
				this.pNode.name = name;
				this.callChangeNotifiers("nameChanged");
			},

			// const return value
			getName : function () {
				return this.pNode.name;
			},

			setMatrix : function (matrix) {
				utils.assert(matrix.length === 9);
				this.pNode.matrix = mat3(matrix);
				this.callChangeNotifiers("matrixChanged");
			},

			getMatrix : function () {
				return this.pNode.matrix.clone();
			},

			/**
			 * Get matrix relative to some ancestor node.
			 */
			getMatrixRelativeTo : function (ancestor, result0) {
				var parent = this.getParent(),
					result = mat3.identity(result0);

				if (this === ancestor) {
					return result;
				}

				mat3.multiply(this.getMatrix(), result, result);

				// otherwise repeat for parent
				if (parent) {
					return mat3.multiply(parent.getMatrixRelativeTo(ancestor), result, result);
				}

				if (ancestor === null) {
					return result;
				}

				// or throw an error when there are no more parents to try
				throw new Error("getMatrixRelativeTo(): ancestor argument was not an ancestor.");
			},

			// Accepts "In Front", "Add", "Screen"
			setBlendMode : function (blendMode) {
				if (this.pNode.blendMode !== blendMode) {
					this.pNode.blendMode = blendMode;
					this.callChangeNotifiers("blendModeChanged");
				}
			},

			setClipped : function (bClipped) {
				if (this.pNode.bClipped !== bClipped) {
					this.pNode.bClipped = bClipped;
					this.callChangeNotifiers("clippingChanged");
				}
			},

			// const return value
			getBlendMode : function () {
				return this.pNode.blendMode;
			},

			// const return value
			getClipped : function () {
				return this.pNode.bClipped;
			},

			setParentCanWarpMe : function (parentCanWarpMe) {
				this.pNode.parentCanWarpMe = parentCanWarpMe;
				this.callChangeNotifiers("parentCanWarpMeChanged");
			},

			getParentCanWarpMe : function () {
				return this.pNode.parentCanWarpMe;
			},

			removeChangeNotifiers: function () {
				this.removeAllChangeNotifiers();
			},
	
			// TODO: Rename/unify with geometry component.
			// This function is called automatically by Zoot's C++ Stage
			// to install methods needed for some simulation methods.
			addSimulationComponent : function (component) {
				// pointer to that component: keep name in sync with NewRigDynamics() in Stage.cpp
				this.pMesh = component;

				this.getMeshGeometry = component.getMeshGeometry.bind(component);
				this.getMeshTopology = component.getMeshTopology.bind(component);
				this.getMeshWeights = component.getMeshWeights.bind(component);
				this.getWarpGeometry = component.getWarpGeometry.bind(component);
			},

			// This function is called automatically by Zoot's C++ Stage
			// to install methods needed for some inspection of node art.
			installGeometryComponent : function (native) {
				utils.assert(native.getBounds, "installGeometryComponent: expected object with 'getBounds' function.");
				utils.assert(native.getGeometryOutline, "installGeometryComponent: expected object with 'getGeometryOutline' function.");
				utils.assert(native.init, "installGeometryComponent: expected object with 'init' function.");
				this.privateGeometry.cpp = native;
			},

			// getBounds() returns bounds of node's art as [left, top, width, height] array.
			// the bounds for container will be a union of all of its children
			getBounds : function () {
				var native = this.privateGeometry.cpp;
				return native.getBounds(0);
			},

			getMeshBounds : function () {
				var native = this.privateGeometry.cpp;
				return native.getBounds(1);
			},

			getGeometryOutline : function () {
				var native = this.privateGeometry.cpp,
					outline = [];

				native.getGeometryOutline(outline);

				// TODO: Should this be moved to the C++ side?
				if (outline.length <= 0) {
					// construct outline from bounding box
					var ltwh = this.getMeshBounds();
					// with desugaring semantics:
					// let [l, t, w, h] = this.getMeshBounds();

					outline[0] = [ ltwh[0], ltwh[1] ];
					outline[1] = [ ltwh[0], ltwh[1] + ltwh[3]];
					outline[2] = [ ltwh[0] + ltwh[2], ltwh[1] + ltwh[3] ];
					outline[3] = [ ltwh[0] + ltwh[2], ltwh[1] ];
				}

				return outline;
			},

			initGeometryComponent : function () {
				var native = this.privateGeometry.cpp;

				native.init();
			},

			// Supported Change Notifier types:
			// "nameChanged"
			// "blendModeChanged"
			// "matrixChanged"
			// "parentChanged"
			// "visibleChanged"
			// "childAdded"
			// "childRemoved"

			clone : function (clone_children, other) {
				var result = other;

				if (result) {
					// init
					Node.call(result);
				} else {
					// alloc and init
					result = new Node();
				}

				utils.clone(true, result.pNode, this.pNode);

				// clone node state
				VisibleItem_clone.call(this, clone_children, result);

				delete result.pNode.parent;
				result.pNode.matrix = this.pNode.matrix.clone();
				delete result.pNode.changeNotifiers;
				result.setParent(undefined);

				return result;
			}
		}
	);

	return Node;
});
